''' Module for adding a GUI interface (via tkinter) to FlashLogs.

Authorship:
duvall@wustl.edu
Siteman Cancer Center,
Washington University School of Medicine
02/2025

'''

##FIXME
# - copy LOGGER from pkg scope [?]

##TODO
# - Implement less-hacky versions of show_*_details()
# - Clean up outdated items since migrating to manual tracking

#REMINDERS:
# Traceable variable types from tkinter are as follows:
# - BooleanVar
# - IntVar
# - DoubleVar
# - StringVar


## IMPORTS
import math, io, contextlib, pathlib
import logging as log
import tkinter as tk
from tkinter import font, simpledialog, filedialog, messagebox
from tkinter.simpledialog import askinteger
from functools import partial, partialmethod
# from ttk...
# this package
from flashlogs import *
# external
from tabulate import tabulate as tab
# # settings
# # log.basicConfig(level = log.WARN)
# # log.basicConfig(level = log.INFO)
# log.basicConfig(level = log.DEBUG)


## CLASSES

# TKVars -- shorthands for traceable tkinter var types
class TKVars():
    ''' Shorthands for traceable tkinter var types\n
    '''
    # values
    B = tk.BooleanVar
    I = tk.IntVar
    D = tk.DoubleVar
    S = tk.StringVar
V = TKVars


# GuiLogAnalyzer
class GuiLogAnalyzer(LogAnalyzer):
    ''' Like LogAnalyzer, but with a GUI.\n
    '''

    def __init__(self, FILE = ''):
        flashlogs.GUI = True
        super().__init__(FILE)
        self.steplist = []
        self.root = tk.Tk()
        self.root.bind('<Control-w>', self.close_all)
        self.root.bind('<Escape>', self.close_all)
        self.root.bind('<Return>', self.handle_return)
        self.defaultFont = font.nametofont('TkDefaultFont')
        self.defaultFont.configure(family = standard_font[0], size = standard_font[1])
        self.root.widgets = []
        self.frames = {}
        self.guisteps = []
        w =  800 # width for the Tk root
        h =  800 # height for the Tk root
        ws = self.root.winfo_screenwidth() # width of the screen
        hs = self.root.winfo_screenheight() # height of the screen
        # x = (ws/2) - (w/2) # single monitor, centered
        x = 3*(ws/4) - (w/2) # single monitor, offset right
        # x = (ws/2) - 3*(w/2) # dual monitors
        y = (hs/2) - (h/2)
        self.root.geometry('%dx%d+%d+%d' % (w, h, x, y))
        self.root.title('FlashLogs LogAnalyzer')
        self.recno_var = V.S(master = self.root, value = None)
        self.recresults_var = V.S(master = self.root, value = None)

        # header
        l_heading = tk.Label(self.root, text = 'LogAnalyzer', justify = 'center', font = header_font)

        # frames
        f_la = tk.Frame(self.root)
        f_record_hdr = tk.Frame(self.root)
        f_record = tk.Frame(self.root)
        f_actions_hdr = tk.Frame(self.root)
        f_actions = tk.Frame(self.root)
        f_controls = tk.Frame(self.root)
        f_close = tk.Frame(self.root)
        self.frames['log_analyzer'] = f_la
        self.frames['records_hdr'] = f_record_hdr
        self.frames['records'] = f_record
        self.frames['actions_hdr'] = f_actions_hdr
        self.frames['controls'] = f_controls 
        self.frames['actions'] = f_actions
        self.frames['close'] = f_close

        # session info
        l_session_hdr = tk.Label(f_la, text = 'FLASH Session Summary', font = subheader_font, justify = 'left')
        l_session = tk.Label(f_la, text = self.session_info(), font = mono_font, justify = 'left')
        l_session_hdr.pack()
        l_session.pack()
        # b_session_details = tk.Button(f_la, text = 'Show Session Details', font = button_font, width = 16, command = self.show_session_details)
        # b_session_details.pack()
        b_start = tk.Button(f_la, text = 'START PLOTS', font = button_font, width = 16, command = lambda: show_first_record(self))
        b_start.pack(pady = 16)

        # record info #TODO: convert to TKVar
        self.recno_var.set(value = f'Current Delivery Record: {self.current_record.record_no:3d}')
        self.recresults_var = V.S(master = self.root, value = self.current_record.gui_info())
        self.get_delivery_record(0)
        l_record_hdr = tk.Label(f_record_hdr, textvariable = self.recno_var, font = subheader_font, justify = 'left')
        l_record = tk.Label(f_record, textvariable = self.recresults_var, font = mono_font, justify = 'left')
        l_record_hdr.pack()
        l_record.pack()
        # b_record_details = tk.Button(f_record, text = 'Show Record Details', font = button_font, width = 16, command = self.show_record_details)
        # b_record_details.pack()

        # prev/go/next buttons
        b_prev = tk.Button(f_controls, text = '<- Prev', font = button_font, command = self.gui_go_prev_record)
        b_prev.pack(side = tk.LEFT, padx = 8)
        b_go = tk.Button(f_controls, text = 'Go to Record...', font = button_font, command = self.gui_ask_record)
        b_go.pack(side = tk.LEFT, padx = 8)
        b_next = tk.Button(f_controls, text = 'Next ->', font = button_font, command = self.gui_go_next_record)
        b_next.pack(side = tk.RIGHT, padx = 8)

        # actions: save and profile functions
        b_save_plot = tk.Button(f_actions, text = 'Save Current DR Plot', font = button_font, command = self.save_plot)
        b_save_all_plots = tk.Button(f_actions, text = 'SAVE ALL PLOTS', font = button_font, command = self.gui_save_all_plots)
        b_profile = tk.Button(f_actions, text = 'Show Profile Plot', font = button_font, command = self.profile)
        b_save_profile = tk.Button(f_actions, text = 'Save Profile Plot', font = button_font, command = self.gui_save_profile)
        b_save_plot.grid(row = 0, column = 0, padx = 4, pady = 4)
        b_save_all_plots.grid(row = 0, column = 1, padx = 4, pady = 4)
        b_profile.grid(row = 1, column = 0, padx = 4, pady = 4)
        b_save_profile.grid(row = 1, column = 1, padx = 4, pady = 4)

        # close button
        b_close = tk.Button(f_close, text = 'Close', font = button_font, width = 8, command = self.root.destroy)
        b_close.pack()
        self.b_close = b_close

        # pack widgets
        l_heading.pack(side = tk.TOP, pady = 8)
        f_la.pack(pady = 8)
        f_record_hdr.pack(pady = 8)
        f_record.pack(pady = 4)
        f_controls.pack(pady = 4)
        f_actions_hdr.pack(pady = 8)
        f_actions.pack(pady = 4)
        f_close.pack(side = tk.BOTTOM, pady = 8)

        # start the GUI event loop
        self.root.update()
        self.root.update_idletasks()
        self.root.mainloop()

    # session_info #TODO: add directory
    def session_info(self):
        ses = [ [ 'Logfile', os.path.basename(self.logfile) ],
                [ 'Logfile Timestamp', self.log_timestamp ],
                [ 'Total Delivery Records', self.total_delivery_records ] ]
        return tab(ses, tablefmt = 'simple_outline')
                # [ 'Analysis Timestamp', self.analysis_timestamp ],

    # close_all
    def close_all(self, _):
        self.root.destroy()

    # handle_return
    def handle_return(self, _):
        focused = self.root.focus_get()
        if isinstance(focused, tk.Button):
            focused.invoke()

    # show_session_details
    def show_session_details(self):
        log.debug('TEST DEBUG -- show_session_details')
        captured_summary = io.StringIO()
        with contextlib.redirect_stdout(captured_summary):
            self.session_info()
        print(self.session_info())

    # show_record_details
    def show_record_details(self):
        log.debug('TEST DEBUG -- show_record_details')
        captured_summary = io.StringIO()
        with contextlib.redirect_stdout(captured_summary):
            self.current_record.info(True)
        log.debug(captured_summary.getvalue())

    # show_current_record
    def show_current_record(self):
        super().show_record(self.current_record.record_no)

    # gui_ask_record
    def gui_ask_record(self):
        recno = askinteger(title = 'Record Selection', prompt = 'Which delivery record?')
        if recno == None:
            return
        valid_records = [_.record_no for _ in self.all_delivery_records]
        if not recno in valid_records:
            messagebox.showerror(title = 'Invalid Record Request', message = 'Requested delivery record not found; please enter a valid delivery-record number.')
            self.gui_ask_record()
        else:
            self.show_record(recno)

    # gui_go_prev_record
    def gui_go_prev_record(self):
        if not self.go_prev_record():
            messagebox.showinfo(title = 'Go Previous', message = 'This is the first record.')

    # gui_go_next_record
    def gui_go_next_record(self):
        if not self.go_next_record():
            messagebox.showinfo(title = 'Go Next', message = 'This is the last record.')

    # gui_save_all_plots
    def gui_save_all_plots(self):
        self.save_all_plots()
        plotdir = os.path.basename(self.plot_absdir) 
        archive_name = plotdir + '.zip'
        msg = f'Done!\n\nIndividual plots saved under\n`{os.path.basename(self.plot_absdir)+os.sep}`.\n\n'
        msg += f'Plot archive saved in this directory as `{archive_name}`.'
        plots_saved = messagebox.showinfo(title = 'Save All Plots', message = msg)

    # gui_save_profile
    def gui_save_profile(self):
        self.save_profile()
        profile_name = f'{os.path.basename(self.datarun)}_profile.png'
        msg = f'Done! Profile plot saved as\n`{profile_name}`.'
        profile_saved = messagebox.showinfo(title = 'Save Profile', message = msg)


## FUNCTIONS

# callbacks:
# update_bool_label
def update_bool_label(var, label, _1 = None, _2 = None, _3 = None): # var = tk.BooleanVar, label = tk.Label
    if var.get():
        label.config(text = 'YES')
        label.config(bg = 'green', fg = 'white')
    else:
        label.config(text = ' NO')
        label.config(bg = 'red', fg = 'white')
    return None
# show_first_record
def show_first_record(gui_log_analyzer):
    first_recno = gui_log_analyzer.all_delivery_records[0].record_no
    gui_log_analyzer.show_record(first_recno)
    mgr = plt.get_current_fig_manager()
    mgr.window.geometry('640x522+300+100') #//HC//
    return None


## DATA

# fonts
if IS_UNIX:
    serif_font_family = 'DejaVu Serif'
    sans_font_family = 'DejaVu Sans'
    mono_font_family = 'DejaVu Sans Mono'
    font_sizes = (20, 16, 14, 14)
else:
    serif_font_family = 'Georgia'
    sans_font_family = 'Calibri'
    mono_font_family = 'Cascadia Code'
    font_sizes = (18, 14, 12, 12)
header_font = (serif_font_family, font_sizes[0], font.BOLD)
subheader_font = (serif_font_family, font_sizes[1], font.BOLD)
button_font = (sans_font_family, font_sizes[2])
standard_font = (sans_font_family, font_sizes[2])
mono_font = (mono_font_family, font_sizes[3])


## all pau!   )
